<?php
/**
 * BuddyPress taxonomy functions.
 *
 * Most BuddyPress taxonomy functions are wrappers for their WordPress counterparts.
 * Because BuddyPress can be activated in various ways in a network environment, we
 * must switch to the root blog before using the WP functions.
 *
 * @package BuddyPress
 * @subpackage Core
 * @since 2.2.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Returns default BuddyPress taxonomies.
 *
 * @since 7.0.0
 *
 * @return array The BuddyPress default taxonomies.
 */
function bp_get_default_taxonomies() {
	$taxonomies = array(
		// Member Type.
		bp_get_member_type_tax_name() => array(
			'object'    => 'user',
			'component' => 'members',
			'args'      => bp_get_member_type_tax_args(),
		)
	);

	/**
	 * This filter should only be used by built-in BuddyPress Components.
	 *
	 * @since 7.0.0
	 *
	 * @param array $taxonomies The taxonomy arguments used for WordPress registration.
	 */
	return apply_filters( 'bp_get_default_taxonomies', $taxonomies );
}

/**
 * Register our default taxonomies.
 *
 * @since 2.2.0
 */
function yz_register_default_taxonomies() {
	$taxonomies = bp_get_default_taxonomies();

	foreach ( $taxonomies as $taxonomy_name => $taxonomy_params ) {
		if ( ! isset( $taxonomy_params['object'] ) || ! isset( $taxonomy_params['args'] ) ) {
			continue;
		}

		register_taxonomy(
			$taxonomy_name,
			$taxonomy_params['object'],
			$taxonomy_params['args']
		);
	}
}
add_action( 'bp_register_taxonomies', 'yz_register_default_taxonomies' );

/**
 * Add a new taxonomy term to the database.
 *
 * @since 7.0.0
 *
 * @param string $term     The BP term name to add.
 * @param string $taxonomy The BP taxonomy to which to add the BP term.
 * @param array  $args {
 *     Optional. Array of arguments for inserting a BP term.
 *     @type string $description The term description. Default empty string.
 *     @type string $slug        The term slug to use. Default empty string.
 *     @type array  $metas       The term metas to add. Default empty array.
 * }
 * @return array|WP_Error An array containing the `term_id` and `term_taxonomy_id`,
 *                        WP_Error otherwise.
 */
function bp_insert_term( $term, $taxonomy = '', $args = array() ) {
	if ( ! taxonomy_exists( $taxonomy ) ) {
		return new WP_Error( 'invalid_taxonomy', __( 'Invalid taxonomy.', 'buddypress' ) );
	}

	$site_id = bp_get_taxonomy_term_site_id( $taxonomy );

	$switched = false;
	if ( $site_id !== get_current_blog_id() ) {
		switch_to_blog( $site_id );
		bp_register_taxonomies();
		$switched = true;
	}

	$term_metas = array();
	if ( isset( $args['metas'] ) ) {
		$term_metas = (array) $args['metas'];
		unset( $args['metas'] );
	}

	/**
	 * Fires before a BP Term is added to the database.
	 *
	 * @since 7.0.0
	 *
	 * @param string $term     The BP term name to add.
	 * @param string $taxonomy The BP taxonomy to which to add the term.
	 * @param array  $args     Array of arguments for inserting a BP term.
	 */
	do_action( 'bp_before_insert_term', $term, $taxonomy, $args );

	$tt_id = wp_insert_term( $term, $taxonomy, $args );

	if ( is_wp_error( $tt_id ) ) {
		return $tt_id;
	}

	$term_id = reset( $tt_id );

	if ( $term_metas ) {
		bp_update_type_metadata( $term_id, $taxonomy, $term_metas );
	}

	if ( $switched ) {
		restore_current_blog();
	}

	/**
	 * Fires when taxonomy terms have been set on BuddyPress objects.
	 *
	 * @since 7.0.0
	 *
	 * @param array  $tt_ids    An array containing the `term_id` and `term_taxonomy_id`.
	 * @param string $taxonomy  Taxonomy name.
	 * @param array  $term_metas The term metadata.
	 */
	do_action( 'bp_insert_term', $tt_id, $taxonomy, $term_metas );

	return $tt_id;
}

/**
 * Get taxonomy BP Terms from the database.
 *
 * @since 7.0.0
 *
 * @param array  $args {
 *     Array of arguments to query BP Terms.
 *     @see `get_terms()` for full description of arguments in case of a member type.
 * }
 * @return array The list of terms matching arguments.
 */
function bp_get_terms( $args = array() ) {
	$args = bp_parse_args(
		$args,
		array(
			'taxonomy'   => '',
			'number'     => '',
			'hide_empty' => false,
		),
		'get_terms'
	);

	if ( ! $args['taxonomy'] ) {
		return array();
	}

	$site_id = bp_get_taxonomy_term_site_id( $args['taxonomy'] );

	$switched = false;
	if ( $site_id !== get_current_blog_id() ) {
		switch_to_blog( $site_id );
		bp_register_taxonomies();
		$switched = true;
	}

	$terms = get_terms( $args );

	if ( $switched ) {
		restore_current_blog();
	}

	/**
	 * Filter here to modify the BP Terms found into the database.
	 *
	 * @since 7.0.0
	 *
	 * @param array $terms The list of terms matching arguments.
	 * @param array $args  Array of arguments used to query BP Terms.
	 */
	return apply_filters(
		'bp_get_terms',
		$terms,
		$args
	);
}

/**
 * Deletes a BP Term.
 *
 * @since 7.0.0
 *
 * @param int     $term_id  The BP Term ID. Required.
 * @param string  $taxonomy The BP Taxonomy Name. Required.
 * @return bool|WP_Error True on success, WP_Error on failure.
 */
function bp_delete_term( $term_id = 0, $taxonomy = '' ) {
	if ( ! $term_id || ! $taxonomy ) {
		return new WP_Error( 'missing_arguments', __( 'Sorry, the term ID and the taxonomy are required arguments.', 'buddypress' ) );
	}

	$site_id = bp_get_taxonomy_term_site_id( $taxonomy );

	$switched = false;
	if ( $site_id !== get_current_blog_id() ) {
		switch_to_blog( $site_id );
		bp_register_taxonomies();
		$switched = true;
	}

	/**
	 * Fires before a BP Term is deleted from the database.
	 *
	 * @since 7.0.0
	 *
	 * @param int    $term_id  The BP Term ID.
	 * @param string $taxonomy The BP Taxonomy Name.
	 */
	do_action( 'bp_before_delete_term', $term_id, $taxonomy );

	$deleted = wp_delete_term( $term_id, $taxonomy );

	if ( $switched ) {
		restore_current_blog();
	}

	if ( is_wp_error( $deleted ) ) {
		return $deleted;
	}

	if ( false === $deleted ) {
		return new WP_Error( 'inexistant_term', __( 'Sorry, the term does not exist.', 'buddypress' ) );
	}

	if ( 0 === $deleted ) {
		return new WP_Error( 'default_term', __( 'Sorry, the default term cannot be deleted.', 'buddypress' ) );
	}

	/**
	 * Fires once a BP Term has been deleted from the database.
	 *
	 * @since 7.0.0
	 *
	 * @param boolean $deleted True.
	 * @param int     $term_id  The deleted BP Term ID.
	 * @param string  $taxonomy The BP Taxonomy Name of the deleted BP Term ID.
	 */
	do_action( 'bp_delete_term', $deleted, $term_id, $taxonomy );

	return $deleted;
}

/**
 * Update a list of metadata for a given type ID and a given taxonomy.
 *
 * @since 7.0.0
 *
 * @param  integer $type_id    The database ID of the BP Type.
 * @param  string  $taxonomy   The BP Type taxonomy.
 * @param  array   $type_metas An associative array (meta_key=>meta_value).
 * @return boolean             False on failure. True otherwise.
 */
function bp_update_type_metadata( $type_id = 0, $taxonomy = '', $type_metas = array() ) {
	if ( ! $type_id || ! $taxonomy || ! is_array( $type_metas ) ) {
		return false;
	}

	foreach ( $type_metas as $meta_key => $meta_value ) {
		if ( ! registered_meta_key_exists( 'term', $meta_key, $taxonomy ) ) {
			continue;
		}

		update_term_meta( $type_id, $meta_key, $meta_value );
	}

	return true;
}

function bp_get_taxonomy_types( $taxonomy = '', $types = array() ) {
	if ( ! $taxonomy ) {
		return $types;
	}

	$db_types = wp_cache_get( $taxonomy, 'bp_object_terms' );

	if ( ! $db_types ) {
		$terms = bp_get_terms(
			array(
				'taxonomy' => $taxonomy,
			)
		);

		if ( ! is_array( $terms ) || ! $terms ) {
			return $types;
		}

		$type_metadata = array_keys( get_registered_meta_keys( 'term', $taxonomy ) );

		foreach ( $terms as $term ) {
			$type_name                      = $term->name;
			$db_types[ $type_name ]         = new stdClass();
			$db_types[ $type_name ]->db_id  = $term->term_id;
			$db_types[ $type_name ]->labels = array();
			$db_types[ $type_name ]->name   = $type_name;

			if ( $type_metadata ) {
				foreach ( $type_metadata as $meta_key ) {
					$type_key = str_replace( 'bp_type_', '', $meta_key );
					if ( in_array( $type_key, array( 'name', 'singular_name' ), true ) ) {
						$db_types[ $type_name ]->labels[ $type_key ] = get_term_meta( $term->term_id, $meta_key, true );
					} else {
						$db_types[ $type_name ]->{$type_key} = get_term_meta( $term->term_id, $meta_key, true );
					}
				}

				if ( ! empty( $db_types[ $type_name ]->has_directory ) && empty( $db_types[ $type_name ]->directory_slug ) ) {
					$db_types[ $type_name ]->directory_slug = $term->slug;
				}
			}
		}

		wp_cache_set( $taxonomy, $db_types, 'bp_object_terms' );
	}

	if ( is_array( $db_types ) ) {
		foreach ( $db_types as $db_type_name => $db_type ) {
			// Override props of registered by code types if customized by the admun user.
			if ( isset( $types[ $db_type_name ] ) && isset( $types[ $db_type_name ]->code ) && $types[ $db_type_name ]->code ) {
				// Merge Labels.
				if ( $db_type->labels ) {
					foreach ( $db_type->labels as $key_label => $value_label ) {
						if ( '' !== $value_label ) {
							$types[ $db_type_name ]->labels[ $key_label ] = $value_label;
						}
					}
				}

				// Merge other properties.
				foreach ( get_object_vars( $types[ $db_type_name ] ) as $key_prop => $value_prop ) {
					if ( 'labels' === $key_prop || 'name' === $key_prop ) {
						continue;
					}

					if ( isset( $db_type->{$key_prop} ) && '' !== $db_type->{$key_prop} ) {
						$types[ $db_type_name  ]->{$key_prop} = $db_type->{$key_prop};
					}
				}

				unset( $db_types[ $db_type_name ] );
			}
		}
	}

	return array_merge( $types, (array) $db_types );
}
