<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// QUERYING
//================================================================================

/**
 * Add new query variables in the list of public query variables, 
 * so that we can use it in our custom URLs.
 * https://codex.wordpress.org/Function_Reference/get_query_var
 */
function add_public_query_vars_for_membership_requests( $qvars ) {
    // Back-end request listings
    $qvars[] = "membership_request_meta_query";
    // Front-end request listings
    //$qvars[] = "viewing";
    return $qvars;
}
add_filter( 'query_vars', 'add_public_query_vars_for_membership_requests' );

/**
 * Manipulate query before quering 
 * and map 'membership_request_meta_query' custom query argument 
 * to the 'meta_query' WordPress's query argument.
 */
function membership_request_pre_get_posts( $query ) {

    // if( function_exists('get_current_screen') && get_current_screen()->parent_base == 'edit' && is_admin() && $query->is_main_query() ) {
    //      //$query->query_vars['suppress_filters'] = true;
    //  }

    /**
     * If we're viewing the administration panel, 
     * and we're quering for "membership-request" custom post type, 
     * and there's a "membership_request_meta_query" custom query argument present, 
     * and its value is not empty.
     */
    if( is_admin() && get_query_var( 'post_type' ) == 'membership-request' && ! empty( get_query_var( 'membership_request_meta_query' ) ) && ! empty( $query->query_vars['membership_request_meta_query'][0]['value'] ) && $query->is_main_query() ) {

        // Map 'membership_request_meta_query' to 'meta_query'
        $query->set( 'meta_query', get_query_var( 'membership_request_meta_query' ) );

    }

}
add_action( 'pre_get_posts', 'membership_request_pre_get_posts' );

//================================================================================
// LISTINGS SCREEN (edit.php) CUSTOMIZATION - COLUMNS, FILTERS etc.
//================================================================================

/**
 * Customize views @edit.php?post_type=membership-request
 * More @ https://codex.wordpress.org/Plugin_API/Filter_Reference/views_edit-post
 */
function remove_views_from_membership_requests($views) {  
    unset($views['mine']);
    return $views;
}
add_filter('views_edit-membership-request', 'remove_views_from_membership_requests');

/**
 * Register columns for the back-end listings of the "membership-request" custom post type.
 * https://codex.wordpress.org/Plugin_API/Filter_Reference/manage_$post_type_posts_columns
 */
function register_backend_membership_request_columns($columns)
{
    return [
        'cb' => '<input type="checkbox" />',
        'title' => __('Title', 'erua'),
        'applicant' => __('Applicant', 'erua'),
        'email' => __('Email', 'erua'),
        'status' => __('Status', 'erua'),
        'date' => __('Date', 'erua'),
    ];
}
add_filter('manage_membership-request_posts_columns', 'register_backend_membership_request_columns');

/**
 * Create the content of the custom columns
 * that were configured with register_backend_membership_request_columns()
 * http://justintadlock.com/archives/2011/06/27/custom-columns-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/253640/adding-custom-columns-to-custom-post-types
 * https://wpsmackdown.com/easy-filter-posts-wordpress-admin/
 * https://codex.wordpress.org/WordPress_Query_Vars
 * https://wordpress.stackexchange.com/questions/212519/filter-by-custom-field-in-custom-post-type-on-admin-page
 */
function manage_membership_request_backend_custom_columns($column, $post_id)
{
    global $post;

    switch ($column) {

        case 'applicant':

            $author_id = $post->post_author;
            $author = get_the_author_meta('display_name', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author
                );
            }
            else {
                if ( get_field('field_6554ffe7792ec', $post_id) && get_field('field_6554ffea792ed', $post_id) ) {
                    $applicant_name = get_field('field_6554ffe7792ec', $post_id);
                    $applicant_email = get_field('field_6554ffea792ed', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'membership_request_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'eruahaus_membership_request_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'eruahaus_membership_request_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_name . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

        case 'email':

            $author_id = $post->post_author;
            $author_email = get_the_author_meta('user_email', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author_email
                );
            }
            else {
                if ( get_field('field_6554ffe7792ec', $post_id) && get_field('field_6554ffea792ed', $post_id) ) {
                    $applicant_name = get_field('field_6554ffe7792ec', $post_id);
                    $applicant_email = get_field('field_6554ffea792ed', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'membership_request_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'eruahaus_membership_request_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'eruahaus_membership_request_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_email . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

		/* If displaying the "status" column. */
        case 'status':

            /* Get the statuses for the post. */
            $terms = get_the_terms($post_id, 'membership-request-status');

            /* If no terms were found, output a default message. */
            if ( empty($terms) ) {
                echo __('Unknown', 'erua');
            }
            /* If there is a status */
            else {
                $out = [];
                foreach ($terms as $term) {
                    // "Open"
                    if ($term->term_id == 150) {
                        $class_name = 'request-status open-status';
                    }
                    // "Solved"
                    else if ($term->term_id == 154) {
                        $class_name = 'request-status approved-status';
                    }
                    // "Closed"
                    else if ($term->term_id == 151) {
                        $class_name = 'request-status closed-status';
                    }
                    // Pending
                    else if ($term->term_id == 153) {
                        $class_name = 'request-status pending-status';
                    }
                    // In Progress
                    else if ($term->term_id == 152) {
                        $class_name = 'request-status inprogress-status';
                    }
                    else {
                        $class_name = 'request-status generic-status';
                    }

                    $out[] = sprintf('<a href="%s" class="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'membership-request-status' => $term->slug], 'edit.php')),
                        $class_name,
                        esc_html(sanitize_term_field('name', $term->name, $term->term_id, 'membership-request-status', 'display'))
                    );
                }
                echo join(', ', $out);
            }

            break;

        /* Just break out of the switch statement for everything else. */
        default:
            break;
    }
}
add_action('manage_membership-request_posts_custom_column', 'manage_membership_request_backend_custom_columns', 10, 2);

/**
 * Add filters based on custom taxonomies,
 * for the membership request listings (@ edit.php).
 * https://generatewp.com/filtering-posts-by-taxonomies-in-the-dashboard/
 */
function create_backend_filters_for_membership_requests($post_type, $which)
{

    // Apply this only on a specific post type
    if ('membership-request' === $post_type) {

        /**
         * A list of taxonomy slugs to filter by
         * Note: Private and draft posts will be displayed 
         * but won't be counted in the taxonomy's terms.
         */
        $taxonomies = ['membership-request-status'];

        foreach ($taxonomies as $taxonomy_slug) {

            // Retrieve taxonomy data
            $taxonomy_obj = get_taxonomy($taxonomy_slug);
            $taxonomy_name = $taxonomy_obj->labels->name;
            $taxonomy_is_hierarchical = $taxonomy_obj->hierarchical;

            if ( $taxonomy_is_hierarchical ) {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );

                    $top_term_id = $top_level_term->term_id;

                    $second_level_terms = get_terms( array(
                        'taxonomy' => $taxonomy_slug, # you could also use $taxonomy as defined in the first lines
                        'child_of' => $top_term_id,
                        'parent' => $top_term_id, # disable this line to see more child elements (child-child-child-terms)
                        'hide_empty' => false,
                    ) );
                    foreach ($second_level_terms as $second_level_term) {

                        $query = get_posts(
                            [
                                'post_type' => $post_type,
                                'fields' => 'ids',
                                'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                                'numberposts' => -1,
                                'tax_query' => [
                                    [
                                        'taxonomy' => $taxonomy_slug,
                                        'field' => 'slug',
                                        'terms' => $second_level_term->slug,
                                        'operator' => 'IN'
                                    ]
                                ]
                            ]
                        );
                        wp_reset_postdata();
                        $count = count($query);

                        printf(
                            '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                            $second_level_term->slug,
                            ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $second_level_term->slug)) ? ' selected="selected"' : ''),
                            '&nbsp;&nbsp;&nbsp;&nbsp;' . $second_level_term->name,
                            $count//$second_level_term->count
                        );
                    }
                }
                echo '</select>';
            }
            else {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('Show All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );
                }
                echo '</select>';
            }
        }

        // membership_request_meta_query[0][key]=associated_event_id&membership_request_meta_query[0][value]=2&membership_request_meta_query[0][compare]=LIKE
        //var_dump($_GET);
        //print_r($_GET['membership_request_meta_query'][0]['value']);
        //echo $query->request; // see the sql query
        // https://sudarmuthu.com/blog/get-the-sql-query-generated-by-wordpress/
        //echo '<br /><br /><br />' . $GLOBALS['wp_query']->request . '<br /><br /><br />'; // see the sql query

    }

}
add_action('restrict_manage_posts', 'create_backend_filters_for_membership_requests', 10, 2);

//================================================================================
// META BOXES
//================================================================================

/**
 * Display Request status meta box.
 * This function is utilized by register_taxonomy()
 * 
 * https://sudarmuthu.com/blog/creating-single-select-wordpress-taxonomies/
 * https://gist.github.com/sudar/e2f97e1455b30dc71d07
 */
function membership_request_status_meta_box( $post ) {
	$terms = get_terms( 'membership-request-status', [ 'hide_empty' => false ] );
	$post  = get_post();
	//$status = wp_get_object_terms( $post->ID, 'membership-request-status', [ 'orderby' => 'term_id', 'order' => 'ASC' ] );
    $status = get_the_terms( $post->ID, 'membership-request-status' );
    $name  = '';
    $id = '';
    if ( ! is_wp_error( $status ) ) {
    	if ( isset( $status[0] ) && isset( $status[0]->name ) ) {
            $name = $status[0]->name;
            $id = $status[0]->term_id;
	    }
    }
    // echo __('<strong><u>Note:</u></strong> Show a note, <strong>here</strong> if you want.', 'erua') . '<br /><br />';
	foreach ( $terms as $term ) {
    ?>
		<label title='<?php esc_attr_e( $term->name ); ?>'>
		    <input type="radio" name="<?php esc_attr_e('tax_input[membership-request-status][]'); ?>" value="<?php esc_attr_e( $term->term_id ); ?>" <?php checked( $term->term_id, $id ); ?>>
			<span><?php esc_html_e( $term->name ); ?></span>
		</label><br>
    <?php
    }
}
/**
 * Display an error message at the top of the post edit screen explaining that a taxonomy is required.
 *
 * Doing this prevents users from getting confused when their new posts aren't published, as we
 * require a valid custom taxonomy.
 *
 * @param WP_Post The current post object.
 */
function membership_request_show_required_field_error_msg( $post ) {
	if ( 'membership-request' === get_post_type( $post ) && 'auto-draft' !== get_post_status( $post ) ) {

        //$status = wp_get_object_terms( $post->ID, 'membership-request-status', [ 'orderby' => 'term_id', 'order' => 'ASC' ] );
        $status = get_the_terms( $post->ID, 'membership-request-status' );

        if ( is_wp_error( $status ) || empty( $status ) ) {
			printf(
				'<div class="error below-h2"><p>%s</p></div>',
				esc_html__( 'Please select a status for this Membership Request.' )
			);
        }

	}
}
// Unfortunately, 'admin_notices' puts this too high on the edit screen
add_action( 'edit_form_top', 'membership_request_show_required_field_error_msg' );

//================================================================================
// BACK-END FIELD HANDLING
//================================================================================

/**
 * Load field values.
 *
 * https://www.advancedcustomfields.com/resources/acf-load_value/
 */
function membership_request_load_applicant_name( $value, $post_id, $field ) {

    $user_id = get_field('eruahaus_membership_request_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->display_name ) {
        $value = $user->display_name;
        // $new_title = $user->display_name . ' (' . $user->user_email . ')';
        // wp_update_post( array( 'post_title' => $new_title ) );
    }

    return $value;
}
add_filter('acf/load_value/key=field_6554ffe7792ec', 'membership_request_load_applicant_name', 10, 3); # Applicant's Name

function membership_request_load_applicant_email( $value, $post_id, $field ) {

    $user_id = get_field('eruahaus_membership_request_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->user_email ) {
        $value = $user->user_email;
        $new_title = $user->display_name . ' (' . $user->user_email . ')';
        wp_update_post( array( 'post_title' => $new_title ) );
    }

    return $value;
}
add_filter('acf/load_value/key=field_6554ffea792ed', 'membership_request_load_applicant_email', 10, 3); # Applicant's E-mail

function membership_request_load_applicant_profile_url( $value, $post_id, $field ) {

    if ( function_exists('bp_is_active') ) {

        $user_id = get_field('eruahaus_membership_request_registered_user_id', $post_id);

        if ( !$user_id ) {
            return $value;
        }

        $user_profile_url = bbp_get_user_profile_url( (int) $user_id );

        if ( $value != $user_profile_url ) {
            $value = $user_profile_url;
        }
    }

    return $value;
}
add_filter('acf/load_value/key=field_655508ced6e48', 'membership_request_load_applicant_profile_url', 10, 3); # Applicant's Profile URL

function membership_request_render_applicant_profile_url( $field ) {
    // echo '<pre>';
    // var_dump($field);
    // echo '</pre>';
    if ( $field['value'] ) {
        echo '<a href="'.$field['value'].'" target="_blank" style="display: inline-block; background-color: aliceblue; padding: 5px; border-radius: 0 0 5px 5px; text-decoration: none;">'.__('Visit profile', 'erua').'</a>';
    }
}
add_action( 'acf/render_field/key=field_655508ced6e48', 'membership_request_render_applicant_profile_url', 10, 1 );

//================================================================================
// DISABLING STUFF
//================================================================================

/**
 * Make an ACF field readonly to prevent editing.
 * https://www.advancedcustomfields.com/resources/acf-load_field/
 * https://support.advancedcustomfields.com/forums/topic/read-only-field-2/
 */
function membership_request_acf_make_field_readonly( $field ) {
	
    // $field['choices'] = array(
    //     'custom' => 'My Custom Choice'
    // );
    //$field['disabled'] = 1;
    $field['readonly'] = true;

    return $field;
    
}
add_filter('acf/load_field/key=field_6554ffe7792ec', 'eruahaus_project_acf_make_field_readonly'); # Applicant's Name
add_filter('acf/load_field/key=field_6554fff8792ee', 'eruahaus_project_acf_make_field_readonly'); # Applicant's User ID
//add_filter('acf/load_field/key=field_611fc5e72cc79', 'membership_request_acf_make_field_readonly'); # Unique ID

/**
 * Disable various fields at a Membership Request's editing screen.
 */
function erua_disable_membership_request_fields() {
    global $pagenow;

    // If we are at the back-office, at a request post.
    if ( is_admin() && !( defined( 'DOING_AJAX' ) && DOING_AJAX ) && ($pagenow == 'post.php' || $pagenow == 'post-new.php') && get_post_type() == 'membership-request' ) {
    ?>
        <script>
        jQuery(document).ready(function() {
            // Disable ACF's date picker at request's editing screen.
            jQuery(".hasDatepicker").prop('disabled', 'disabled');
            // Disable various Publishing actions
            jQuery("body.wp-admin #misc-publishing-actions .edit-timestamp").css({display: "none"});

            <?php //if ( is_user_logged_in() && !current_user_can('administrator') ) : ?>
            // Disable Title
            //jQuery("#title").prop('disabled', true);
            // Disable various Publishing actions
            jQuery("body.wp-admin #misc-publishing-actions #major-publishing-actions").css({display: "none"});

            // Change "Publish" text acf-field-5a8be31e0f56a
            jQuery("body.wp-admin #submitdiv h2.hndle").html("<span><?php _e('Application Actions', 'erua'); ?></span>");
            //jQuery("#ml_box").remove();
            //jQuery("#major-publishing-actions").remove();
            //jQuery("#minor-publishing-actions").remove();
            //jQuery("#misc-publishing-actions .misc-pub-post-status").remove();
            //jQuery("#visibility").remove();
            // Disable addition and removal of attachments
            //jQuery("body.wp-admin .acf-file-uploader .acf-icon.-cancel").css({display: "none"});
            //jQuery("body.wp-admin .hide-if-value a[data-name='add']").css({display: "none"});
            //jQuery("body.wp-admin .show-if-value a[data-name='edit']").css({display: "none"});

            <?php //endif; ?>
        });
        </script>
    <?php
    }

}
add_action('admin_print_footer_scripts', 'erua_disable_membership_request_fields');

//================================================================================
// BUBBLE NOTIFICATIONS
//================================================================================

/**
 * Notification "bubble" with pending/open Membership Requests count
 */
add_action('admin_menu', function () {
    global $menu;
    //$count_posts = wp_count_posts('membership-request');
    //$count = $count_posts->pending;

    // View all requests with an awaiting nature (pending, waiting for x's reply etc.) of status
    $requests_query = get_posts(
        [
            'post_type' => 'membership-request',
            'fields' => 'ids',
            //'post_status' => 'any', // Published, Pending etc.
            //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash'],
            //'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'numberposts' => -1,
            // All requests that are not approved or closed i.e. pending or open
            'tax_query' => [
                [
                    'taxonomy' => 'membership-request-status',
                    'field' => 'slug',
                    'terms' => ['closed', 'approved'],
                    'operator' => 'NOT IN',
                ],
            ],
        ]
    );
    if (!empty($requests_query)) wp_reset_postdata();
    $count = count($requests_query);
    /* echo '<pre>';
    var_dump($requests_query);
    echo '</pre>';
    die(); */

    $menu_item = wp_list_filter(
        $menu,
        [2 => 'edit.php?post_type=membership-request'] // 2 is the position of an array item which contains URL, it will always be 2!
    );
    if ( !empty($menu_item) && $count > 0 ) {
        $menu_item_position = key($menu_item); // get the array key (position) of the element
        $menu[$menu_item_position][0] .= ' <span class="awaiting-mod" title="' . $count . ' Requests require actions">' . $count . '</span>';
    }
});
