<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Get the host domain.
 *
 * @return string $domain The site's domain
 */
function get_host_domain() {
    $urlparts = parse_url(trim(site_url()));
    $domain = $urlparts['host'];
    // parse_url returns the host (ex. example.com) as the path when no scheme is provided in the input url
    $domain = trim($urlparts['host'] ? $urlparts['host'] : array_shift(explode('/', $urlparts['path'], 2)));
    // keep only the naked domain if there's a www prefix
    // if (substr($temp, 0, 4) === 'www.') {
    //     $domain = str_replace('www.', '', $temp);
    // }
    $domain = str_replace('www.', '', $domain);

    return $domain;
}

//================================================================================
// STRING OPERATIONS
//================================================================================

/**
 * Get X number of words from a string.
 *
 * https://www.sitepoint.com/community/t/how-to-echo-only-a-certain-number-of-words/2972/4
 * https://www.w3resource.com/php-exercises/php-string-exercise-24.php
 *
 * @param string $string The full string that we want to cut.
 * @param int $words The number of words that we want to extract.
 *
 * @return string $reduced_string Returns an array of whatever
 */
function get_words($string, $words) {
    /**
     * We explode the string into an array of words,
     * then slice it and reduce it to a new array of
     * only the number of words that we want, and then
     * we implode and join all word array elements into a final string.
     */
    $reduced_string = implode(' ', array_slice(explode(' ', $string), 0, $words));
    return $reduced_string;
}

//================================================================================
// DATE OPERATIONS
//================================================================================

/**
 * Validate a date format
 * https://stackoverflow.com/questions/3964972/validate-that-input-is-in-this-time-format-hhmm
 *
 * @param string $date
 * @param string $format
 * @return boolean
 */
function isValidDate(string $date, string $format = 'Y-m-d'): bool {
    $dateObj = DateTime::createFromFormat($format, $date);
    return $dateObj && $dateObj->format($format) == $date;
}

//================================================================================
// ARRAY OPERATIONS
//================================================================================

/**
 * Insert a value or key/value pair after a specific key in an array.
 *
 * https://gist.github.com/wpscholar/0deadce1bbfa4adb4e4c
 */
function associative_array_insert_after(array &$array, $key, array $new ) {
    $keys  = array_keys( $array );
    $index = array_search( $key, $keys, true  );
    $pos   = false === $index ? count( $array ) : $index + 1;
    
    $array = array_slice($array, 0, $pos, true) + $new + array_slice($array, $pos, count($array) - 1, true);
    //return $array;
}

//================================================================================
// FILEBIRD-RELATED
//================================================================================

/**
 * Move a file attachment that has been uploaded to the Media Library,
 * to a specified FileBird virtual folder.
 *
 * https://ninjateam.gitbook.io/filebird/developer-zone/functions
 */
function move_attachment_to_filebird_folder( $attachment_id, $filebird_folder_id ) {

    if ( !is_numeric($attachment_id) || !is_numeric($filebird_folder_id) ) {
        return false;
    }

    $attachment_id = (int) $attachment_id;
    $filebird_folder_id = (int) $filebird_folder_id;

    if ( !is_int($attachment_id) || $attachment_id <= 0 ) {
        return false;
    }

    require_once WP_PLUGIN_DIR . '/filebird-pro/includes/Model/Folder.php';

    //$post_ids = [ 1, 2, 3 ];
    $post_ids = [$attachment_id];

    $folder = "FileBird\Model\Folder"::findById( $filebird_folder_id, 'parent' );

    if ( empty($folder) ) {
        return false;
    }

    "FileBird\Model\Folder"::setFoldersForPosts( $post_ids, $filebird_folder_id );
}

/**
 * Check if a FileBird folder exists.
 */
function filebird_folder_exists($filebird_folder_id) {

    if ( ! is_numeric($filebird_folder_id ) ) {
        return false;
    }

    require_once WP_PLUGIN_DIR . '/filebird-pro/includes/Model/Folder.php';

    $filebird_folder_id = (int) $filebird_folder_id;

    $folder = "FileBird\Model\Folder"::findById( $filebird_folder_id, 'parent' );

    if ( empty($folder) ) {
        return false;
    }
    else {
        return true;
    }
}

/**
 * Get all attachment IDs in a FileBird folder.
 */
function filebird_folder_get_attachment_ids($filebird_folder_id) {

    if ( ! is_numeric($filebird_folder_id ) ) {
        return false;
    }

    // if ( is_plugin_active('filebird-pro/filebird.php') ) {
    // }

    require_once WP_PLUGIN_DIR . '/filebird-pro/includes/Classes/Helpers.php';
    require_once WP_PLUGIN_DIR . '/filebird-pro/includes/Model/Folder.php';

    $filebird_folder_id = (int) $filebird_folder_id;

    $folder = "FileBird\Model\Folder"::findById( $filebird_folder_id, 'parent' );

    if ( empty($folder) ) {
        return false;
    }

    $attachment_ids = "FileBird\Classes\Helpers"::getAttachmentIdsByFolderId( $filebird_folder_id );

    if ( empty($attachment_ids) ) {
        return false;
    }
    else {
        return $attachment_ids;
    }
}
